import re

from pipelines.prompta.rag.meta_data import META_DATA


class Verb:

    def __init__(self, identifier: str, code_pattern: str, symbol_pattern: str) -> None:
        self.identifier = identifier
        self.code_pattern = re.compile(code_pattern)
        self.code_format = code_pattern.replace("""['\\"]([\\w\\d_]+)['\\"]""", '\"{}\"').replace('\\', '')
        self.symbol_pattern = symbol_pattern
        self.symbol_format = re.sub(r'\(.*\)', '{}', symbol_pattern).replace('\\', '')

    def to_symbol(self, code):
        match = re.match(self.code_pattern, code)
        noun = None if not match else match.groups()[0]
        return self.symbol_format.format(noun)
    
    def to_code(self, symbol):
        match = re.match(self.symbol_pattern, symbol)
        noun = None if not match else match.groups()[0]
        return self.code_format.format(noun)
    
    def accept(self, symbol: str) -> bool:
        match = re.match(self.symbol_pattern, symbol)
        return match is not None


CraftVerb = Verb("craftItem", r"await craftItem\(bot, ['\"]([\w\d_]+)['\"]\);", r"craft(.*)")
MineVerb = Verb("mineBlock", r"await mineBlock\(bot, ['\"]([\w\d_]+)['\"]\);", r"mine(.*)")
PlaceVerb = Verb("placeItem", r"await placeItem\(bot, ['\"]([\w\d_]+)['\"]\);", r"place(.*)")
ExploreVerb = Verb("exploreUntilFind", r"await exploreUntilFind\(bot, ['\"]([\w\d_]+)['\"]\);", r"exploreUntilFind(.*)")
GivePlacedItemBackVerb = Verb("givePlacedItemBack", r"await givePlacedItemBack\(bot, ['\"]([\w\d_]+)['\"]\);", r"givePlaced(.*)Back")
EquipVerb = Verb("equip", r"await equip\(bot, ['\"]([\w\d_]+)['\"]\);", r"equip(.*)")
KillMobVerb = Verb("killMob", r"await killMob\(bot, ['\"]([\w\d_]+)['\"]\);", r"kill(.*)")
SmeltVerb = Verb("smeltItem", r"await smeltItem\(bot, ['\"]([\w\d_]+)['\"]\);", r"smelt(.*)")
GetItemFromChestVerb = Verb("getItemFromChest", r"await getItemFromChest\(bot, ['\"]([\w\d_]+)['\"]\);", r"get(.*)FromChest")
DepositItemIntoChestVerb = Verb("depositItemIntoChest", r"await depositItemIntoChest\(bot, ['\"]([\w\d_]+)['\"]\);", r"deposit(.*)IntoChest")
ConsumeVerb = Verb("consume", r"await bot.consume\(['\"]([\w\d_]+)['\"]\);", r"consume(.*)")
FishVerb = Verb("fish", r"await bot.fish\(\);", r"fish")
SleepVerb = Verb("sleep", r"await sleep\(\);", r"sleep")
ActivateBlockVerb = Verb("activateBlock", r"await bot.activateBlock\(['\"]([\w\d_]+)['\"]\);", r"activateBlock(.*)")
ActivateItemVerb = Verb("activateItem", r"await bot.activateItem\(['\"]([\w\d_]+)['\"]\);", r"activateItem(.*)")
UseToolOnEntityVerb = Verb("useToolOnEntity", r"await bot.useToolOnEntity\(['\"]([\w\d_]+)['\"]\);", r"useToolOnEntity(.*)")

VERBS = [
    CraftVerb,
    MineVerb,
    PlaceVerb,
    ExploreVerb,
    GivePlacedItemBackVerb,
    EquipVerb,
    KillMobVerb,
    SmeltVerb,
    GetItemFromChestVerb,
    DepositItemIntoChestVerb,
    ConsumeVerb,
    FishVerb,
    SleepVerb,
    ActivateBlockVerb,
    ActivateItemVerb,
    UseToolOnEntityVerb
]